import numpy as np
from OpenGL.GL import *
import OpenGL.GL.shaders

# pip install pyopengl


def loadShader(t: int, source: str) -> tuple[int, str]:
    # Create
    shader = glCreateShader(t)
    # Load source
    glShaderSource(shader, source)
    # Compile
    glCompileShader(shader)
    # Check status
    if not glGetShaderiv(shader, GL_COMPILE_STATUS):
        return None, "Failed to compile shader\n%s" + glGetShaderInfoLog(shader).decode()

    return shader, None


def loadProgram(vertexSource: str, fragmentSource: str) -> tuple[int, str]:
    # Vertex shader
    vertexShader, err = loadShader(GL_VERTEX_SHADER, vertexSource)
    if err != None:
        return 0, err

    # Fragment shader
    fragmentShader, err = loadShader(GL_FRAGMENT_SHADER, fragmentSource)
    if err != None:
        return 0, err

    # Shader program
    shaderProgram = glCreateProgram()
    glAttachShader(shaderProgram, vertexShader)
    glAttachShader(shaderProgram, fragmentShader)
    glLinkProgram(shaderProgram)

    # Check status
    if not glGetProgramiv(shaderProgram, GL_LINK_STATUS):
        log = glGetShaderInfoLog(shaderProgram)
        glDeleteProgram(shaderProgram)
        return None, "failed to  link program\n%s" + log

    # Clean no longer needed shaders
    glDeleteShader(vertexShader)
    glDeleteShader(fragmentShader)

    return shaderProgram, None


class YUYV:
    def Draw(self, width: int, height: int, data: bytes) -> str:
        ptr = ctypes.c_void_p(ctypes.addressof(ctypes.c_char_p.from_buffer(data)))

        glClear(GL_COLOR_BUFFER_BIT)

        # Use the program object
        glUseProgram(self.shaderProgram)

        # Load the vertex position
        vertices = np.array(
            [
                # Position Texcoords
                - 1.0, +1.0, 0.0, 0.0,  # Top - left
                - 1.0, -1.0, 0.0, 1.0,  # Bottom - left
                + 1.0, -1.0, 1.0, 1.0,  # Bottom - right
                + 1.0, +1.0, 1.0, 0.0,  # Top - right
            ], dtype=np.float32
        )
        glBindBuffer(GL_ARRAY_BUFFER, self.vbo)
        glBufferData(GL_ARRAY_BUFFER, vertices, GL_STATIC_DRAW)
        glVertexAttribPointer(self.posAttrib, 2, GL_FLOAT, GL_FALSE, 4 * 4, ctypes.c_void_p(0 * 4))
        glEnableVertexAttribArray(self.posAttrib)

        # Load the texture coordinate
        indices = np.array(
            [
                0, 1, 2,
                0, 2, 3,
            ], dtype=np.uint32
        )
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, self.ebo)
        glBufferData(GL_ELEMENT_ARRAY_BUFFER, indices, GL_STATIC_DRAW)
        glVertexAttribPointer(self.texAttrib, 2, GL_FLOAT, GL_FALSE, 4 * 4, ctypes.c_void_p(2 * 4))
        glEnableVertexAttribArray(self.texAttrib)

        # Y
        glActiveTexture(GL_TEXTURE1)
        glUniform1i(self.yTexture, 1)
        glBindTexture(GL_TEXTURE_2D, self.yTextureName)
        glTexImage2D(GL_TEXTURE_2D, 0, GL_RG, width, height, 0, GL_RG, GL_UNSIGNED_BYTE, ptr)
        # texture filtering params
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR)
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR)
        # texture wrapping params
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE)
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE)

        # UV
        glActiveTexture(GL_TEXTURE2)
        glUniform1i(self.uvTexture, 2)
        glBindTexture(GL_TEXTURE_2D, self.uvTextureName)
        glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, width / 2, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, ptr)
        # texture filtering params
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR)
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR)
        # texture wrapping params
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE)
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE)

        # Draw
        glDrawElements(GL_TRIANGLES, len(indices), GL_UNSIGNED_INT, None)

        return None


# New creates a yuv422 OpenGL surface
def New() -> tuple[YUYV, str]:
    s = YUYV()

    # Vertex shader
    vertexSource = """
    attribute vec4 a_position;
    attribute vec2 a_texCoord;
    varying vec2 v_texCoord;
    void main() {
        gl_Position = a_position;
        v_texCoord = a_texCoord;
    }
    """

    # Fragment shader
    fragmentSource = """
    varying vec2 v_texCoord;
    uniform sampler2D y_texture;
    uniform sampler2D uv_texture;

    void main(void){
            float r, g, b, y, u, v;
            y  = texture2D(y_texture,  v_texCoord).r;
            u  = texture2D(uv_texture, v_texCoord).g - 0.5;
            v  = texture2D(uv_texture, v_texCoord).a - 0.5;

            r = y + 1.13983 * v;
            g = y - 0.39465 * u - 0.58060 * v;
            b = y + 2.03211 * u;

        gl_FragColor = vec4(r, g, b, 1.0);
    }
    """

    # Create and compile shaders
    s.shaderProgram, err = loadProgram(vertexSource, fragmentSource)
    if err != None:
        return None, err

    # Specify the layout of the vertex data
    s.posAttrib = glGetAttribLocation(s.shaderProgram, "a_position")
    s.texAttrib = glGetAttribLocation(s.shaderProgram, "a_texCoord")

    # Vertex Buffer object
    s.vbo = glGenBuffers(1)
    # Element array object
    s.ebo = glGenBuffers(1)

    # Y Texture
    glEnable(GL_TEXTURE_2D)
    s.yTexture = glGetUniformLocation(s.shaderProgram, "y_texture")
    s.yTextureName = glGenTextures(1)

    # UV Texture
    glEnable(GL_TEXTURE_2D)
    s.uvTexture = glGetUniformLocation(s.shaderProgram, "uv_texture")
    s.uvTextureName = glGenTextures(1)

    # Black background
    glClearColor(0.0, 0.0, 0.0, 0.0)

    return s, None
