#
# WARNING: on Ubuntu, this application should be run as admin
#

import threading
import queue
import time
import glfw
from OpenGL.GL import *
import yuyv

if __package__:
    from ... import lt
else:
    import sys
    sys.path.append("../..")
    import lt

# pip install glfw pyopengl numpy

# Display fifo
display = queue.Queue(1)


class FrameGrabber(threading.Thread):
    def __init__(self, *args, **kwargs):
        super(FrameGrabber, self).__init__(*args, **kwargs)
        self._stop_event = threading.Event()
        self._exception = None

    def run(self):
        try:
            client = lt.Client()

            # Find first active input video source
            sourceURL = ""
            sources = {
                "lt100:/0/cvbs-in/0",
                "lt100:/0/svideo-in/0",
                "lt100:/0/dvi-in/0", "lt100:/0/dvi-in/1",
                "lt100:/0/sdi-in/0", "lt100:/0/sdi-in/1",
            }
            for source in sources:
                input, err = client.Get(source)
                if err != None:
                    exit(err)
                if input['video']['signal'] == 'locked':
                    sourceURL = source
                    print('{:s} found'.format(sourceURL))
                    break

            # If no active input video source found, use a default one
            if sourceURL == "":
                print("no active input video source found")
                sourceURL = "lt100:/0/sdi-in/0"

            # Create worker
            resp, err = client.Post(sourceURL+"/data", {'media': "video/yuyv"})
            if not lt.ErrorIs(err, lt.ErrRedirect):
                exit(err)
            workerURL = lt.RedirectLocation(err)

            # Loop
            cnt = 0
            prevTS = 0
            missed = 0
            dropped = 0

            start = time.time()
            while not self._stop_event.is_set():
                # FPS
                duration = time.time() - start
                if duration > 1:
                    print("%.2f FPS - missed %d - dropped %d" % (cnt / duration, missed, dropped))
                    start = time.time()
                    cnt = 0

                # Fetch worker update
                worker, err = client.Get(workerURL)
                if err != None:
                    raise Exception(err)

                # Packet data
                if len(worker['packets']) == 0:
                    exit("packet not found")
                packet = worker['packets'][0]

                # Packet metadata
                meta = packet['meta']

                # Count missed packet
                ts = packet['timestamp']
                if prevTS != 0:
                    inc = int(round((ts - prevTS) * meta['framerate'] / 1_000_000))
                    missed += inc - 1
                prevTS = ts

                try:
                    # Send packet to display thread
                    display.put(packet, False)
                except:
                    # Count dropped packet
                    dropped += 1

                # Loop counter
                cnt += 1

        except Exception as e:
            self._exception = e
        finally:
            self._stop_event.set()

    def stop(self):
        self._stop_event.set()

    def rethrow(self):
        if self._exception:
            raise self._exception


def main():
    # Turn-on the frame grabber thread
    frame_grabber = FrameGrabber()

    # Initialize the GLFW library
    if not glfw.init():
        exit("Could not initialize GLFW")

    # Create a windowed mode window and its OpenGL context
    window = glfw.create_window(640, 360, "Python player", None, None)
    if not window:
        glfw.terminate()
        exit()

    # Make the window's context current
    glfw.make_context_current(window)

    # OpenGL surface to render to
    surface, err = yuyv.New()
    if err != None:
        glfw.terminate()
        exit(err)

    # Start frame grabber thread
    frame_grabber.start()

    # Loop until the user closes the window
    while not glfw.window_should_close(window):
        try:
            # Check if an error occurred in the FrameGrabber thread
            frame_grabber.rethrow()

            packet = display.get()

            # OpenGL draw buffer
            surface.Draw(packet['meta']['size'][0], packet['meta']['size'][1], packet['data'])

            # Window size
            w, h = glfw.get_framebuffer_size(window)
            glViewport(0, 0, w, h)

            # Poll for and process events
            glfw.poll_events()
            # Swap front and back buffers
            glfw.swap_buffers(window)
        except Exception as e:
            print(f"An error occurred: {e}")
            break
        except KeyboardInterrupt:
            print("KeyboardInterrupt has been caught.")
            break

    # Stop the FrameGrabber thread
    if frame_grabber is not None:
        frame_grabber.stop()
        frame_grabber.join()

    # Done
    glfw.terminate()


if __name__ == "__main__":
    main()
