package main

import (
	"encoding/json"
	"errors"
	"fmt"
	"log"
	"math"
	"runtime"
	"time"

	lt "lt/client/go"
	"lt/client/go/example/video_player/surface"

	"github.com/go-gl/gl/v4.1-core/gl"
	"github.com/go-gl/glfw/v3.3/glfw"
)

// Display fifo
var play = make(chan *lt.Packet, 1)

func main() {
	// Fetch video frames thread
	go func() {
		var client lt.Client
		defer client.Close()

		// Find first active input video source
		var sourceURL string
		sources := []string{
			"lt100:/0/cvbs-in/0",
			"lt100:/0/svideo-in/0",
			"lt100:/0/dvi-in/0", "lt100:/0/dvi-in/1",
			"lt100:/0/sdi-in/0", "lt100:/0/sdi-in/1",
		}
		for _, source := range sources {
			var input lt.Input
			if err := client.Get(source, &input); err != nil {
				log.Fatal(err)
			}
			if input.Video.Signal == "locked" {
				sourceURL = source
				fmt.Printf("%s found\n", sourceURL)
				break
			}
		}

		// If no active input video source found, use a default one
		if sourceURL == "" {
			fmt.Println("no active input video source found")
			sourceURL = "lt100:/0/sdi-in/0"
		}

		// Select media type
		media := "video/yuyv" // or "video/nv12", "video/rgb", "video/rgba" ...

		// Create a worker to capture YUYV video frames
		err := client.Post(sourceURL+"/data", lt.VideoDataWorker{Media: media}, nil)
		if !errors.Is(err, lt.ErrRedirect) {
			log.Fatal("worker creation failed:", err)
		}
		workerURL := lt.RedirectLocation(err)

		// Loop
		var prevTS int64
		var missed, dropped int
		start := time.Now()
		for cnt := 0; ; cnt++ {
			// FPS
			end := time.Now()
			duration := end.Sub(start).Seconds()
			if duration > 1 {
				fmt.Printf("%.2f FPS - missed %d - dropped %d\n", float64(cnt)/duration, missed, dropped)
				start = end
				cnt = 0
			}

			// Fetch worker response
			var worker lt.Worker
			if err := client.Get(workerURL, &worker); err != nil {
				log.Fatal(err)
			}

			// Packet data
			if len(worker.Packets) == 0 {
				log.Fatal("worker packet: not found")
			}
			packet := worker.Packets[0]

			// Packet metadata
			var meta lt.VideoMetadata
			if err := json.Unmarshal(packet.Meta, &meta); err != nil {
				log.Fatal("worker packet metadata:", err)
			}

			// Count missed packet
			ts := packet.Timestamp
			if prevTS != 0 {
				inc := int(math.Round(float64(ts-prevTS) * meta.Framerate / 1_000_000))
				missed += inc - 1
			}
			prevTS = ts

			select {
			// Send packet to display thread
			case play <- &packet:
			// Count dropped packet
			default:
				dropped += 1
				// Release packet reference
				packet.Close()
			}
		}
	}()

	// Initialize the GL library
	runtime.LockOSThread()
	if err := glfw.Init(); err != nil {
		log.Fatal(err)
	}
	defer glfw.Terminate()

	// Create a windowed mode window and its OpenGL context
	title := "Go player"
	// title, _ := strings.CutSuffix(vidUrl, "/data")
	window, err := glfw.CreateWindow(640, 360, title, nil, nil)
	if err != nil {
		log.Fatal(err)
	}
	window.SetFramebufferSizeCallback(surface.SizeCallback)
	window.SetKeyCallback(surface.KeyCallback)

	// Make the window's context current
	window.MakeContextCurrent()

	// OpenGL
	if err := gl.Init(); err != nil {
		log.Fatal(err)
	}
	fmt.Println("OpenGL version", gl.GoStr(gl.GetString(gl.VERSION)))

	// OpenGL surface to render to
	s, err := surface.NewYUYV()
	if err != nil {
		log.Fatal(err)
	}

	// Render loop
	for !window.ShouldClose() {
		packet := <-play

		// Packet metadata
		var meta lt.VideoMetadata
		if err := json.Unmarshal(packet.Meta, &meta); err != nil {
			log.Fatal("worker packet metadata:", err)
		}

		// OpenGL draw buffer
		s.Draw(meta.Size[0], meta.Size[1], packet.Data)
		// Release packet reference
		packet.Close()

		// Update window size
		w, h := window.GetFramebufferSize()
		gl.Viewport(0, 0, int32(w), int32(h))

		// Swap front and back buffers
		window.SwapBuffers()
		glfw.PollEvents()
	}
}
