#pragma warning disable CA1416 // Disable windows only codes warnings

using System;
using System.Text;
using System.IO;
using System.IO.Pipes;
using System.IO.MemoryMappedFiles;
using System.Threading;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Runtime.InteropServices;

using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System.Runtime.Serialization;

namespace lt
{
    public interface Error
    {
        string ToString();
    }

    class Errors : Error
    {
        public static Error EOF = New("EOF");
        public static Error ErrRedirect = New("redirect");
        public static Error ErrClosed = New("lt100agent not found");
        private string s;
        private Errors(string s)
        {
            this.s = s;
        }
        public static Error New(string s)
        {
            return new Errors(s);
        }
        public static bool Is(Error err, Error target)
        {
            if (err == null || target == null)
            {
                return false;
            }
            return err.ToString().Substring(0, target.ToString().Length).Equals(target.ToString());
        }
        public static string RedirectLocation(Error err)
        {
            if (err == null)
            {
                return "";
            }
            return err.ToString().Substring(10);
        }
        public override string ToString()
        {
            return s;
        }
    }


    //
    // Packet
    //

    [JsonObject]
    public class Packet : IDisposable
    {
        public int track { get; set; } = 0;
        public string media { get; set; } = "";
        public string signal { get; set; } = "";

        [JsonIgnore] public unsafe byte* data = null;
        [JsonIgnore] public int length = 0;
        public JObject meta { get; set; } = null;
        public Int64 timestamp { get; set; } = 0;

        [JsonProperty("data")] private string DataString { get; set; } = "";
        [JsonProperty("handle")] private string Handle { get; set; } = "";
        [JsonProperty("ptr")] private int Ptr { get; set; } = 0;
        [JsonProperty("len")] private int Len { get; set; } = 0;
        [JsonProperty("cap")] private int Cap { get; set; } = 0;
        [JsonProperty("ref")] public string Ref { get; set; } = "";

        private byte[] Data = null;
        private Buffer Buf = null;
        private GCHandle DataHandle;
        private bool disposed = false;
        private Component component = new Component();

        public RoundTripper rt = null;

        [OnDeserialized]
        private void OnDeserialized(StreamingContext context)
        {
            // Decode base64 data field
            if (Data != null)
            {
                Data = Convert.FromBase64String(DataString);
                DataHandle = GCHandle.Alloc(Data, GCHandleType.Pinned);
                unsafe
                {
                    data = (byte*)DataHandle.AddrOfPinnedObject();
                }
                length = Data.Length;
                // Done
                return;
            }

            // Capture shared memory reference
            if (Ref.Length == 0)
            {
                // Done
                return;
            }

            // Load shared memory data
            (Buf, var ok) = SharedBuffers.Load(Handle, Cap);
            if (!ok)
            {
                return;
            }
            unsafe
            {
                data = Buf.Data + Ptr;
            }
            length = Len;

            // Done
            return;
        }

        public void Dispose()
        {
            Dispose(disposing: true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!disposed)
            {
                if (disposing)
                {
                    component.Dispose();
                }
                // Release pinned *data*
                if (DataHandle.IsAllocated)
                {
                    DataHandle.Free();
                }
                // Release shared buffer reference
                if (Ref != null)
                {
                    SharedBuffers.Delete(Handle);
                    if (rt != null)
                    {
                        JObject r;
                        rt.call("DELETE", Ref, null, out r);
                    }
                }
                // Done
                disposed = true;
            }
        }

        ~Packet()
        {
            Dispose(disposing: false);
        }
    }


    //
    // Worker
    //

    [JsonObject]
    public class Worker
    {
        public string name { get; set; } = "";
        public string location { get; set; } = "";

        public Int64 start { get; set; } = 0;
        public Int64 duration { get; set; } = 0;
        public Int64 length { get; set; } = 0;
        public string status { get; set; } = "";

        public Packet[] packets { get; set; }
    }


    //
    // Shared Buffer
    //

    class Buffer : IDisposable
    {
        public int Ref = 0;
        public unsafe byte* Data = null;
        public int Len = 0;

        private MemoryMappedViewAccessor va;
        private bool disposed = false;
        private Component component = new Component();

        public static (Buffer, Error) Map(string name, int size)
        {
            var buf = new Buffer();

            // Open shared memory
            var handle = MemoryMappedFile.OpenExisting(name);
            if (handle == null)
            {
                return (null, Errors.New("could not open shared memory: " + name));
            }
            // Map shared memory
            buf.va = handle.CreateViewAccessor(0, size, MemoryMappedFileAccess.Read);
            if (buf.va == null)
            {
                return (null, Errors.New("could not map shared memory: " + name));
            }
            unsafe
            {
                buf.va.SafeMemoryMappedViewHandle.AcquirePointer(ref buf.Data);
                if (buf.Data == null)
                {
                    return (null, Errors.New("could not acquire pointer: " + name));
                }
                buf.Len = size;
            }
            return (buf, null);
        }

        public void Dispose()
        {
            Dispose(disposing: true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!disposed)
            {
                if (disposing)
                {
                    component.Dispose();
                }
                // Clean up
                va.SafeMemoryMappedViewHandle.ReleasePointer();
                // Done
                disposed = true;
            }
        }

        ~Buffer()
        {
            Dispose(disposing: false);
        }
    }

    static class SharedBuffers
    {
        private static Dictionary<string, Buffer> m = new Dictionary<string, Buffer>();
        private static readonly object mu = new object();

        public static (Buffer, bool) Load(string handle, int size)
        {
            lock (mu)
            {
                // Map shared buffer data
                if (!m.ContainsKey(handle))
                {
                    var (b, err) = Buffer.Map(handle, size);
                    if (err != null)
                    {
                        return (null, false);
                    }
                    m[handle] = b;
                }
                // Increment buffer ref
                m[handle].Ref += 1;
                // Done
                return (m[handle], true);
            }
        }

        public static void Delete(string handle)
        {
            lock (mu)
            {
                if (m.ContainsKey(handle))
                {
                    m[handle].Ref -= 1;
                    if (m[handle].Ref <= 0)
                    {
                        //m.Remove(handle);
                    }
                }

            }
        }
    }


    //
    // Conn
    //

    class pipeConn
    {
        NamedPipeClientStream pipe;

        public Error Dial(string addr)
        {
            // Local pipe
            pipe = new NamedPipeClientStream(".", addr, PipeDirection.InOut);
            if (pipe == null)
            {
                return Errors.New("could not open named pipe: " + addr);
            }
            try
            {
                pipe.Connect(1);
            }
            catch
            {
                return Errors.New("named pipe not found: " + addr);
            }
            pipe.ReadMode = PipeTransmissionMode.Message;
            return null;
        }

        public (int, Error) Read(byte[] p)
        {
            var n = pipe.Read(p, 0, p.Length);
            return (n, null);
        }

        public (int, Error) Write(byte[] p)
        {
            pipe.Write(p, 0, p.Length);
            return (p.Length, null);
        }
    }


    //
    // Client
    //

    public class RoundTripper
    {
        private string scheme = "";
        private pipeConn conn = new pipeConn();
        private byte[] connBuffer = new byte[1024 * 1024 * 16];
        private static readonly object mu = new object();

        private Error open(string addr)
        {
            return conn.Dial(addr);
        }

        public Error call(string method, string url, in JObject body, out JObject response)
        {
            response = default(JObject);

            lock (mu)
            {
                // Validate url
                Uri uri = new Uri(url);
                if (uri.Scheme == "")
                {
                    return Errors.New("url scheme not found");
                }
                if (uri.PathAndQuery == "")
                {
                    return Errors.New("url path not found");
                }
                // Validate connection
                if (scheme != "" && scheme != uri.Scheme)
                {
                    return Errors.New("bad url scheme: " + scheme + "!=" + uri.Scheme);
                }
                if (scheme == "")
                {
                    var err = open(uri.Scheme);
                    if (err != null)
                    {
                        return err;
                    }
                    scheme = uri.Scheme;
                }

                // JSON request
                {
                    var request = new JObject {
                        { "method", method },
                        { "url", url },
                        { "body", body }
                    };

                    var err = encode(request);
                    if (err != null)
                    {
                        return err;
                    }
                }

                // JSON Response
                {
                    var err = decode(out response);
                    if (err != null)
                    {
                        return err;
                    }
                }

                // Check for errors
                {
                    Error err = null;
                    string location = "";
                    if (response.ContainsKey("error"))
                    {
                        err = Errors.New((string)response["error"]);
                    }
                    if (response.ContainsKey("location"))
                    {
                        location = (string)response["location"];
                    }
                    if (Errors.Is(err, Errors.ErrRedirect))
                    {
                        return Errors.New("redirect: " + location);
                    }
                    else if (err != null)
                    {
                        return err;
                    }
                }

                // Done
                return null;
            }
        }

        private Error decode(out JObject j)
        {
            j = null;
            var (n, err) = conn.Read(connBuffer);
            if (err != null)
            {
                return err;
            }
            var data = Encoding.ASCII.GetString(connBuffer[0..n]);
            if (data == "null")
            {
                return null;
            }

            try
            {
                j = JObject.Parse(data);
                return null;
            }
            catch (JsonReaderException e)
            {
                return Errors.New("roundTripper decode error: " + e.Message);
            }
        }

        private Error encode(in JObject j)
        {
            var data = Encoding.UTF8.GetBytes(JsonConvert.SerializeObject(j, Formatting.None));
            var (_, err) = conn.Write(data);
            return err;
        }
    }

    public class Client
    {
        public RoundTripper roundTripper = new RoundTripper();

        public Error Get(string url, out object nullResponse)
        {
            nullResponse = null;

            JObject r;
            return call("GET", url, null, out r);
        }

        public Error Get<Response>(string url, out Response response)
        {
            response = default(Response);

            // Process request
            JObject r;
            var err = call("GET", url, null, out r);
            if (err != null)
            {
                return err;
            }
            // Parse response
            response = r.ToObject<Response>();
            return null;
        }


        public Error Get(string url, out Worker worker)
        {
            worker = default(Worker);

            // Process request
            JObject r;
            var err = call("GET", url, null, out r);
            if (err != null)
            {
                return err;
            }
            // Parse response
            worker = r.ToObject<Worker>();
            foreach (var packet in worker.packets)
            {
                if (packet.Ref != "")
                {
                    packet.rt = roundTripper;
                }
            }
            return null;
        }


        public Error Post(string url, in object nullBody, out object nullResponse)
        {
            nullResponse = null;

            JObject r;
            return call("POST", url, null, out r);
        }

        public Error Post<Body>(string url, in Body body, out object nullResponse)
        {
            nullResponse = null;

            JObject r;
            return call("POST", url, (JObject)JToken.FromObject(body), out r);
        }

        public Error Post<Response>(string url, in object nullBody, out Response response)
        {
            response = default(Response);

            // Process request
            JObject r;
            var err = call("POST", url, null, out r);
            if (err != null)
            {
                return err;
            }
            // Parse response
            response = r.ToObject<Response>();
            return null;
        }

        public Error Post<Body, Response>(string url, in Body body, out Response response)
        {
            response = default(Response);

            // Process request
            JObject r;
            var err = call("POST", url, (JObject)JToken.FromObject(body), out r);
            if (err != null)
            {
                return err;
            }
            // Parse response
            response = r.ToObject<Response>();
            return null;
        }

        public Error Delete(string url)
        {
            JObject r;
            return call("DELETE", url, null, out r);
        }

        private Error call(string method, string url, in JObject body, out JObject response)
        {
            return roundTripper.call(method, url, body, out response);
        }
    }
}