﻿using System;
using System.IO;
using System.Diagnostics;

using System.Text.Json;
using System.Text.Json.Nodes;

using OpenTK;
using OpenTK.Graphics;
using OpenTK.Graphics.OpenGL;

// dotnet add package OpenTK

namespace Player
{

    class Program
    {
        static void Main(string[] args)
        {
            using (Game game = new Game(640, 360, "C# player"))
            {
                game.VSync = OpenTK.VSyncMode.Off;
                game.Run(0);
            }
        }
    }

    class Game : GameWindow
    {
        static void logFatal(string s)
        {
            Console.WriteLine(s);
            System.Environment.Exit(-1);
        }

        public Game(int width, int height, string title) : base(width, height, GraphicsMode.Default, title)
        { }

        lt.Error err;
        lt.Client client = new lt.Client();
        string workerURL;

        long prevTS = 0;
        int cnt = 0;
        int missed = 0;
        Stopwatch timer = Stopwatch.StartNew();

        int shaderProgram;
        int vsID, fsID;
        int texAttrib, posAttrib;
        Int32 yTexture, uvTexture;
        UInt32 yTextureName, uvTextureName;
        int vbo, ebo;

        // Array of our vertex positions
        float[] vertices = new float[] {
                // Position Color         Texcoords
		        (float)-1.0, (float)+1.0, (float)0.0, (float)0.0, // Top-left
		        (float)-1.0, (float)-1.0, (float)0.0, (float)1.0, // Bottom-left
		        (float)+1.0, (float)-1.0, (float)1.0, (float)1.0, // Bottom-right
		        (float)+1.0, (float)+1.0, (float)1.0, (float)0.0, // Top-right
            };

        // Array of our vertex colors
        UInt32[] indices = new UInt32[] {
                0, 1, 2,
                0, 2, 3,
            };

        protected override void OnLoad(EventArgs e)
        {
            shaderProgram = GL.CreateProgram();

            var vertexSource = @"
                attribute vec4 a_position;
                attribute vec2 a_texCoord;
                varying vec2 v_texCoord;
                void main() {
                    gl_Position = a_position;
                    v_texCoord = a_texCoord;
                }
            ";
            loadShader(vertexSource, ShaderType.VertexShader, shaderProgram, out vsID);

            var fragmentSource = @"
                varying vec2 v_texCoord;
                uniform sampler2D y_texture;
                uniform sampler2D uv_texture;

                void main(void){
                    float r, g, b, y, u, v;
                    y = texture2D(y_texture, v_texCoord).r;
                    u = texture2D(uv_texture, v_texCoord).g - 0.5;
                    v = texture2D(uv_texture, v_texCoord).a - 0.5;

                    r = y + 1.13983 * v;
                    g = y - 0.39465 * u - 0.58060 * v;
                    b = y + 2.03211 * u;

                    gl_FragColor = vec4(r, g, b, 1.0);
                }
            ";
            loadShader(fragmentSource, ShaderType.FragmentShader, shaderProgram, out fsID);

            GL.LinkProgram(shaderProgram);
            Console.WriteLine(GL.GetProgramInfoLog(shaderProgram));

            // Specify the layout of the vertex data
            posAttrib = GL.GetAttribLocation(shaderProgram, "a_position");
            texAttrib = GL.GetAttribLocation(shaderProgram, "a_texCoord");

            // Vertex Buffer object
            GL.GenBuffers(1, out vbo);
            // Element array object
            GL.GenBuffers(1, out ebo);

            // Y Texture
            GL.Enable(EnableCap.Texture2D);
            yTexture = GL.GetUniformLocation(shaderProgram, "y_texture");
            GL.GenTextures(1, out yTextureName);

            // UV Texture
            GL.Enable(EnableCap.Texture2D);
            uvTexture = GL.GetUniformLocation(shaderProgram, "uv_texture");
            GL.GenTextures(1, out uvTextureName);

            // Black background
            GL.ClearColor(0, 0, 0, 0);

            // Find first active input video source
            var sourceURL = "";
            string[] sources = new string[] {
                "lt100:/0/cvbs-in/0",
                "lt100:/0/svideo-in/0",
                "lt100:/0/dvi-in/0", "lt100:/0/dvi-in/1",
                "lt100:/0/sdi-in/0", "lt100:/0/sdi-in/1",
            };
            foreach (var source in sources)
            {
                lt.Input input;
                err = client.Get(source, out input);
                if (err != null)
                {
                    logFatal(err.ToString());
                }
                if (input.video.signal == "locked")
                {
                    sourceURL = source;
                    Console.WriteLine("{0} found", sourceURL);
                    break;
                }
            }

            // If no active input video source found, use a default one
            if (sourceURL == "")
            {
                Console.WriteLine("no active input video source found");
                sourceURL = "lt100:/0/sdi-in/0";
            }

            // Create worker
            err = client.Post(sourceURL + "/data", new lt.VideoDataWorker { media = "video/yuyv" }, out _);
            if (!lt.Errors.Is(err, lt.Errors.ErrRedirect))
            {
                logFatal("worker creation failed:" + err);
            }
            workerURL = lt.Errors.RedirectLocation(err);

            base.OnLoad(e);
        }

        void loadShader(String source, ShaderType type, int program, out int address)
        {
            address = GL.CreateShader(type);
            GL.ShaderSource(address, source);
            GL.CompileShader(address);
            GL.AttachShader(program, address);
            Console.WriteLine(GL.GetShaderInfoLog(address));
        }

        protected override void OnRenderFrame(FrameEventArgs e)
        {
            // FPS
            var duration = timer.ElapsedMilliseconds;
            if (duration > 1000)
            {
                Console.WriteLine("{0} FPS - missed {1}", ((float)(cnt * 1000) / duration).ToString("0.00"), missed);
                timer.Restart();
                cnt = 0;
            }

            // Fetch worker response
            lt.Worker worker;
            err = client.Get(workerURL, out worker);
            if (err != null)
            {
                logFatal(err.ToString());
            }

            // Check packet
            if (worker.packets.Length == 0)
            {
                logFatal("worker has no packets");
            }
            var packet = worker.packets[0];
            var meta = packet.meta.ToObject<lt.VideoMetadata>();

            // Count missed packet ?
            var ts = packet.timestamp;
            if (prevTS != 0)
            {
                var inc = (int)(Math.Round((float)(ts - prevTS) * meta.framerate / 1_000_000));
                missed += inc - 1;
            }
            prevTS = ts;

            // Draw
            GL.Clear(ClearBufferMask.ColorBufferBit);

            // Use the program object
            GL.UseProgram(shaderProgram);

            // Load the vertex position
            GL.BindBuffer(BufferTarget.ArrayBuffer, vbo);
            GL.BufferData<float>(BufferTarget.ArrayBuffer, vertices.Length * 4, vertices, BufferUsageHint.StaticDraw);
            GL.VertexAttribPointer(posAttrib, 2, VertexAttribPointerType.Float, false, 4 * 4, 0 * 4);
            GL.EnableVertexAttribArray(posAttrib);

            // Load the texture coordinate
            GL.BindBuffer(BufferTarget.ElementArrayBuffer, ebo);
            GL.BufferData<UInt32>(BufferTarget.ElementArrayBuffer, indices.Length * 4, indices, BufferUsageHint.StaticDraw);
            GL.VertexAttribPointer(texAttrib, 2, VertexAttribPointerType.Float, false, 4 * 4, 2 * 4);
            GL.EnableVertexAttribArray(texAttrib);

            // Y
            GL.ActiveTexture(TextureUnit.Texture1);
            GL.Uniform1(yTexture, 1);
            GL.BindTexture(TextureTarget.Texture2D, yTextureName);
            unsafe
            {
                GL.TexImage2D(
                    TextureTarget.Texture2D,
                    0, PixelInternalFormat.Rg8, meta.size[0], meta.size[1],
                    0, PixelFormat.Rg, PixelType.UnsignedByte, (IntPtr)packet.data
                );
            }
            // Texture filtering params
            GL.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureMinFilter, (int)TextureMinFilter.Linear);
            GL.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureMagFilter, (int)TextureMagFilter.Linear);
            // Texture wrapping params
            GL.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureWrapS, (int)TextureWrapMode.ClampToEdge);
            GL.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureWrapT, (int)TextureWrapMode.ClampToEdge);

            // UV
            GL.ActiveTexture(TextureUnit.Texture2);
            GL.Uniform1(uvTexture, 2);
            GL.BindTexture(TextureTarget.Texture2D, uvTextureName);
            unsafe
            {
                GL.TexImage2D(
                    TextureTarget.Texture2D,
                    0, PixelInternalFormat.Rgba8, meta.size[0] / 2, meta.size[1],
                    0, PixelFormat.Rgba, PixelType.UnsignedByte, (IntPtr)(packet.data)
                );
            }
            // Texture filtering params
            GL.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureMinFilter, (int)TextureMinFilter.Linear);
            GL.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureMagFilter, (int)TextureMagFilter.Linear);
            // Texture wrapping params
            GL.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureWrapS, (int)TextureWrapMode.ClampToEdge);
            GL.TexParameter(TextureTarget.Texture2D, TextureParameterName.TextureWrapT, (int)TextureWrapMode.ClampToEdge);

            // Release packet shared buffer
            packet.Dispose();

            // Draw
            GL.DrawElements(PrimitiveType.Triangles, indices.Length, DrawElementsType.UnsignedInt, 0);
            Context.SwapBuffers();

            cnt++;

            base.OnRenderFrame(e);
        }

        protected override void OnResize(EventArgs e)
        {
            GL.Viewport(0, 0, Width, Height);
            base.OnResize(e);
        }
    }
}
