﻿using System;
using System.IO;

namespace histogram
{
    class Program
    {
        static void logFatal(string s)
        {
            Console.WriteLine(s);
            System.Environment.Exit(-1);
        }

        static void Main(string[] args)
        {
            lt.Error err;
            var client = new lt.Client();

            // Find first active input video source
            var sourceURL = "";
            string[] sources = new string[] {
                "lt100:/0/cvbs-in/0",
                "lt100:/0/svideo-in/0",
                "lt100:/0/dvi-in/0", "lt100:/0/dvi-in/1",
                "lt100:/0/sdi-in/0", "lt100:/0/sdi-in/1",
            };
            foreach (var source in sources)
            {
                lt.Input input;
                err = client.Get(source, out input);
                if (err != null)
                {
                    logFatal(err.ToString());
                }
                if (input.video.signal == "locked")
                {
                    sourceURL = source;
                    break;
                }
            }

            // If no active input video source found, use a default one
            if (sourceURL == "")
            {
                Console.WriteLine("no active input video source found");
                sourceURL = "lt100:/0/sdi-in/0";
            }

            // Create worker
            err = client.Post(sourceURL + "/data", new lt.ImageDataWorker { media = "image/yuv422" }, out _);
            if (!lt.Errors.Is(err, lt.Errors.ErrRedirect))
            {
                logFatal("worker creation failed:" + err);
            }
            var workerURL = lt.Errors.RedirectLocation(err);

            // Fetch worker response
            lt.Worker worker;
            err = client.Get(workerURL, out worker);
            if (err != null)
            {
                logFatal(err.ToString());
            }

            // Check packet
            if (worker.packets.Length == 0)
            {
                logFatal("worker packet: not found");
            }
            var packet = worker.packets[0];

            // Packet metadata
            var meta = packet.meta.ToObject<lt.ImageMetadata>();

            // Print infos
            Console.WriteLine("{0} {1}x{2} {3} bytes", sourceURL, meta.size[0], meta.size[1], packet.length);

            // Create histogram
            var n = meta.size[0] * meta.size[1];
            int[] h = new int[8];
            for (var i = 0; i < n; i++)
            {
                unsafe
                {
                    h[packet.data[i] / 32]++;
                }
            }

            // Print histogram
            Console.WriteLine("");
            Console.WriteLine("  0.. 31: {0:0.#}%", 100 * (float)h[0] / (float)n);
            Console.WriteLine(" 32.. 63: {0:0.#}%", 100 * (float)h[1] / (float)n);
            Console.WriteLine(" 64.. 95: {0:0.#}%", 100 * (float)h[2] / (float)n);
            Console.WriteLine(" 96..127: {0:0.#}%", 100 * (float)h[3] / (float)n);
            Console.WriteLine("128..159: {0:0.#}%", 100 * (float)h[4] / (float)n);
            Console.WriteLine("160..191: {0:0.#}%", 100 * (float)h[5] / (float)n);
            Console.WriteLine("192..223: {0:0.#}%", 100 * (float)h[6] / (float)n);
            Console.WriteLine("224..256: {0:0.#}%", 100 * (float)h[7] / (float)n);

            packet.Dispose();
        }
    }
}
