#ifndef __LT_H__
#define __LT_H__

#include <map>
#include <memory>
#include <mutex>
#include <string>
#include <vector>

// Third party JSON library (MIT License): https://github.com/nlohmann/json
#include "json.hpp"

// LT models
#include "lt_schema.h"

namespace lt {

//
// Errors
//

using error = std::string;
const error ErrEOF = "EOF";
const error ErrRedirect = "redirect";
const error ErrClosed = "lt100agent not found";
inline bool ErrorIs(error err, error target) { return err.find(target) != std::string::npos; };
inline string RedirectLocation(error err) { return err.substr(string("redirect: ").length()); };

//
// Forward declarations
//

class PacketObject;
using Packet = std::shared_ptr<PacketObject>;
class WorkerObject;
using Worker = std::shared_ptr<WorkerObject>;
class bufferObject;
using buffer = std::shared_ptr<bufferObject>;
class roundTripperObject;
using roundTripper = std::shared_ptr<roundTripperObject>;

//
// Packets
//

class PacketObject {
    friend class Client;

public:
    PacketObject(const json& j, error& err);
    ~PacketObject();

    int track = 0;
    string media = "";
    string signal = "";
    int64_t timestamp = 0;

    string ref = "";
    char* data = nullptr;
    int length = 0;
    json meta;

private:
    bytes Data;
    string handle;
    buffer buf;
    roundTripper roundTripper;
};

inline void to_json(json&, const Packet&) { } // Unused

inline void from_json(const json& j, Packet& packet)
{
    error err;
    packet.reset(new PacketObject(j, err));
    if (!err.empty()) {
        throw err;
    }
}

//
// Workers
//

class WorkerObject {
public:
    string name;
    string location;

    int64_t start;
    int64_t duration;
    int length;
    string status;

    std::vector<Packet> packets;
};

NLOHMANN_DEFINE_TYPE_NON_INTRUSIVE(WorkerObject, name, location, start, duration, length, status, packets)

inline void to_json(json&, const Worker&) { } // Unused

inline void from_json(const json& j, Worker& worker)
{
    worker.reset(new WorkerObject());
    j.get_to(*worker);
}

//
// Client
//

class Client {
public:
    Client();

    //
    // GET
    //

    inline error Get(string const& url, json& response)
    {
        return call("GET", url, json(), response);
    };

    inline error Get(string const& url, std::nullptr_t)
    {
        json r;
        return call("GET", url, json(), r);
    };

    template <typename Response>
    inline error Get(string const& url, Response& response)
    {
        // Process request
        json r;
        error err = Get(url, r);
        if (!err.empty()) {
            return err;
        }
        // Parse response
        try {
            r.get_to(response);
        } catch (json::exception& e) {
            return e.what();
        }
        // Done
        return "";
    };

    inline error Get(string const& url, Worker& worker)
    {
        // Process request
        json r;
        error err = Get(url, r);
        if (!err.empty()) {
            return err;
        }
        // Parse response
        try {
            r.get_to(worker);
        } catch (json::exception& e) {
            return e.what();
        }
        // Attach client to packets
        for (auto packet : worker->packets) {
            if (!packet->ref.empty()) {
                packet->roundTripper = roundTripper;
            }
        }
        //  Done
        return "";
    };

    //
    // POST
    //

    inline error Post(string const& url, json const& body, json& response)
    {
        return call("POST", url, body, response);
    };

    inline error Post(string const& url, json const& body, std::nullptr_t)
    {
        json r;
        return call("POST", url, body, r);
    };

    inline error Post(string const& url, std::nullptr_t, json& response)
    {
        return call("POST", url, json(), response);
    };

    inline error Post(string const& url, std::nullptr_t, std::nullptr_t)
    {
        json r;
        return call("POST", url, json(), r);
    };

    template <typename Body, typename Response>
    inline error Post(string const& url, Body const& body, Response& response)
    {
        // Process request
        json b = body, r;
        error err = Post(url, b, r);
        if (!err.empty()) {
            return err;
        }
        // Parse response
        try {
            r.get_to(response);
        } catch (json::exception& e) {
            return e.what();
        }
        // Done
        return "";
    };

    template <typename Body>
    inline error Post(string const& url, Body const& body, std::nullptr_t)
    {
        return Post(url, json(body), nullptr);
    };

    template <typename Response>
    inline error Post(string const& url, std::nullptr_t, Response& response)
    {
        // Process request
        json r;
        error err = Post(url, nullptr, r);
        if (!err.empty()) {
            return err;
        }
        // Parse response
        try {
            r.get_to(response);
        } catch (json::exception& e) {
            return e.what();
        }
        // Done
        return "";
    };

    //
    // DELETE
    //

    inline error Delete(string const& url)
    {
        json r;
        return call("DELETE", url, json(), r);
    };

private:
    error call(string const& method, string const& url, json const& body, json& response);
    roundTripper roundTripper;
};

//
// GET
//

template <typename Response>
inline error Get(string const& url, Response& response) { return Client().Get(url, response); }

inline error Get(string const& url, std::nullptr_t) { return Client().Get(url, nullptr); }

//
// POST
//

template <typename Body, typename Response>
inline error Post(string const& url, Body const& body, Response& response) { return Client().Post(url, body, response); }

template <typename Body>
inline error Post(string const& url, Body const& body, std::nullptr_t) { return Client().Post(url, body, nullptr); }

template <typename Response>
inline error Post(string const& url, std::nullptr_t, Response* response) { return Client().Post(url, nullptr, response); }

inline error Post(string const& url, std::nullptr_t, std::nullptr_t) { return Client().Post(url, nullptr, nullptr); }

//
// DELETE
//

inline error Delete(string const& url) { return Client().Delete(url); }

} // namespace lt
#endif // __LT_H__
