#include "yuyv.h"

YUYV::YUYV()
{
    error err;

    // Vertex shader
    auto vertexSource = "attribute vec4 a_position;"
                        "attribute vec2 a_texCoord;"
                        "varying vec2 v_texCoord;"
                        "void main() {"
                        "    gl_Position = a_position;"
                        "    v_texCoord = a_texCoord;"
                        "}\n";

    // Fragment shader
    auto fragmentSource = "varying vec2 v_texCoord;"
                          "    uniform sampler2D y_texture;"
                          "    uniform sampler2D uv_texture;"
                          "\n"
                          "void main (void){"
                          "    float r, g, b, y, u, v;"
                          "    y = texture2D(y_texture, v_texCoord).r;"
                          "    u = texture2D(uv_texture, v_texCoord).g - 0.5;"
                          "    v = texture2D(uv_texture, v_texCoord).a - 0.5;"
                          "\n"
                          "    r = y + 1.13983*v;"
                          "    g = y - 0.39465*u - 0.58060*v;"
                          "    b = y + 2.03211*u;"
                          "\n"
                          "    gl_FragColor = vec4(r, g, b, 1.0);"
                          "}\n";

    // Create and compile shaders
    err = loadProgram(vertexSource, fragmentSource, &this->shaderProgram);
    if (err != "") {
        this->err = err;
        return;
    }

    // Specify the layout of the vertex data
    this->posAttrib = glGetAttribLocation(this->shaderProgram, "a_position");
    this->texAttrib = glGetAttribLocation(this->shaderProgram, "a_texCoord");

    // Vertex Buffer object
    glGenBuffers(1, &this->vbo);
    // Element array object
    glGenBuffers(1, &this->ebo);

    // Y Texture
    glEnable(GL_TEXTURE_2D);
    this->yTexture = glGetUniformLocation(this->shaderProgram, "y_texture");
    glGenTextures(1, &this->yTextureName);

    // UV Texture
    glEnable(GL_TEXTURE_2D);
    this->uvTexture = glGetUniformLocation(this->shaderProgram, "uv_texture");
    glGenTextures(1, &this->uvTextureName);

    // Black background
    glClearColor(0.0, 0.0, 0.0, 0.0);
}

void YUYV::Draw(int width, int height, char* data)
{
    // Create a Vertex Buffer Object and copy the vertex data to it
    float vertices[] = {
        // Position Texcoords
        -1.0, +1.0, 0.0, 0.0, // Top-left
        -1.0, -1.0, 0.0, 1.0, // Bottom-left
        +1.0, -1.0, 1.0, 1.0, // Bottom-right
        +1.0, +1.0, 1.0, 0.0, // Top-right
    };

    // Create an element array
    uint32_t indices[] = {
        0, 1, 2,
        2, 3, 0
    };

    glClear(GL_COLOR_BUFFER_BIT);

    // Use the program object
    glUseProgram(this->shaderProgram);

    // Load the vertex position
    glBindBuffer(GL_ARRAY_BUFFER, this->vbo);
    glBufferData(GL_ARRAY_BUFFER, sizeof(vertices), vertices, GL_STATIC_DRAW);
    glVertexAttribPointer(this->posAttrib, 2, GL_FLOAT, GL_FALSE, 4 * 4, (void*)(0 * 4));
    glEnableVertexAttribArray(this->posAttrib);

    // Load the texture coordinate
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, this->ebo);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(indices), indices, GL_STATIC_DRAW);
    glVertexAttribPointer(this->texAttrib, 2, GL_FLOAT, GL_FALSE, 4 * 4, (void*)(2 * 4));
    glEnableVertexAttribArray(this->texAttrib);

    // Y
    glActiveTexture(GL_TEXTURE1);
    glUniform1i(this->yTexture, 1);
    glBindTexture(GL_TEXTURE_2D, this->yTextureName);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RG, width, height, 0, GL_RG, GL_UNSIGNED_BYTE, data);
    // texture filtering params
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    // texture wrapping params
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

    // U
    glActiveTexture(GL_TEXTURE2);
    glUniform1i(this->uvTexture, 2);
    glBindTexture(GL_TEXTURE_2D, this->uvTextureName);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, width / 2, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, data);
    // texture filtering params
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    // texture wrapping params
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

    // Draw
    glDrawElements(GL_TRIANGLES, sizeof(indices), GL_UNSIGNED_INT, NULL);
}