#include <chrono>
#include <cmath>
#include <iostream>
#include <stdio.h>

// 0
#include "glad/gl.h"
// 1
#include "glfw/glfw3.h"

#include "../../lt.h"
#include "yuyv.h"

void logFatal(const string err)
{
    std::cout << err << std::endl;
    exit(-1);
}

int main(void)
{
    // Initialize GLFW
    if (!glfwInit())
        exit(EXIT_FAILURE);

    // Create a windowed mode window and its OpenGL context
    GLFWwindow* window = glfwCreateWindow(640, 360, "C++ player", NULL, NULL);
    if (!window) {
        glfwTerminate();
        exit(EXIT_FAILURE);
    }

    // Make the window's context current
    glfwMakeContextCurrent(window);

    // OpenGL
    gladLoadGL(glfwGetProcAddress);
    std::cout << glGetString(GL_VERSION) << "\n";

    // OpenGL surface to render to
    YUYV surface = YUYV();
    if (surface.err != "") {
        logFatal(surface.err);
    }

    // Create lt client
    lt::error err;
    lt::Client client;

    // Find first active input video source
    std::string sourceURL;
    std::vector<string> sources = {
        "lt100:/0/cvbs-in/0",
        "lt100:/0/svideo-in/0",
        "lt100:/0/dvi-in/0", "lt100:/0/dvi-in/1",
        "lt100:/0/sdi-in/0", "lt100:/0/sdi-in/1",
    };
    for (auto& source : sources) {
        lt::Input input;
        err = client.Get(source, input);
        if (!err.empty()) {
            logFatal(err);
        }
        if (input.video.signal == "locked") {
            sourceURL = source;
            printf("%s found\n", sourceURL.c_str());
            break;
        }
    }

    // If no active input video source found, use a default one
    if (sourceURL == "") {
        printf("no active input video source found\n");
        sourceURL = "lt100:/0/sdi-in/0";
    }

    // Create YUYV data stream
    err = client.Post(sourceURL + "/data", lt::VideoDataWorker { "video/yuyv" }, nullptr);
    if (!lt::ErrorIs(err, lt::ErrRedirect)) {
        logFatal("worker creation failed:" + err);
    }

    // Process worker location
    string workerURL = err.substr(string("redirect: ").length());

    int64_t prevTS = 0;
    int cnt = 0, missed = 0;

    auto start = std::chrono::system_clock::now();
    while (!glfwWindowShouldClose(window)) {

        // FPS
        auto end = std::chrono::system_clock::now();
        std::chrono::duration<double> duration = end - start;
        if (duration.count() > 1) {
            printf("%.2f FPS - missed %d\n", double(cnt) / duration.count(), missed);
            start = end;
            cnt = 0;
        }

        // Fetch worker update
        lt::Worker worker;
        err = client.Get(workerURL, worker);
        if (!err.empty()) {
            logFatal(err);
        }

        // Parse stream video packet
        if (worker->packets.empty()) {
            logFatal("video not found");
        }
        lt::Packet packet = worker->packets[0];
        lt::VideoMetadata meta = packet->meta.template get<lt::VideoMetadata>();

        // Count missed packet
        int64_t ts = packet->timestamp;
        if (prevTS != 0) {
            int inc = int(round(double(ts - prevTS) * meta.framerate / 1000000));
            missed += inc - 1;
        }
        prevTS = ts;

        // OpenGL draw buffer
        surface.Draw(meta.size[0], meta.size[1], packet->data);

        // Update window size
        int w, h;
        glfwGetFramebufferSize(window, &w, &h);
        glViewport(0, 0, w, h);

        // Swap front and back buffers
        glfwSwapBuffers(window);
        glfwPollEvents();

        cnt++;
    }

    glfwTerminate();
    exit(EXIT_SUCCESS);
}
