#include "../../lt.h"
#include <iostream>

using string = std::string;

#ifdef _WIN32
#include <direct.h>
#define GET_CURDIR _getcwd
#else
#include <unistd.h>
#define GET_CURDIR getcwd
#endif

void logFatal(lt::error err)
{
    std::cout << err << std::endl;
    exit(-1);
}

int main()
{
    lt::error err;
    lt::Client client;

    // Find first active input video source
    std::string sourceURL;
    std::vector<string> sources = {
        "lt100:/0/cvbs-in/0",
        "lt100:/0/svideo-in/0",
        "lt100:/0/dvi-in/0", "lt100:/0/dvi-in/1",
        "lt100:/0/sdi-in/0", "lt100:/0/sdi-in/1",
    };
    for (auto& source : sources) {
        lt::Input input;
        err = client.Get(source, input);
        if (!err.empty()) {
            logFatal(err);
        }
        if (input.video.signal == "locked") {
            sourceURL = source;
            break;
        }
    }

    // If no active input video source found, use a default one
    if (sourceURL == "") {
        printf("no active input video source found\n");
        sourceURL = "lt100:/0/sdi-in/0";
    }

    // Select an absolute path to a capture directory
    char buffer[1024];
    string wd = string(GET_CURDIR(buffer, sizeof(buffer))); // Current working directory

    // Create worker
    err = client.Post(sourceURL + "/file", lt::ImageFileWorker { "image/jpeg", wd }, nullptr);
    if (!lt::ErrorIs(err, lt::ErrRedirect)) {
        logFatal("worker creation failed:" + err);
    }
    string workerURL = lt::RedirectLocation(err);

    // Fetch worker response
    lt::Worker worker;
    err = client.Get(workerURL, worker);
    if (!err.empty()) {
        logFatal(err);
    }

    // Check packet
    if (worker->packets.empty()) {
        logFatal("worker packet: not found");
    }
    lt::Packet packet = worker->packets[0];

    // Meta
    lt::ImageMetadata meta = packet->meta.template get<lt::ImageMetadata>();

    // Print file information
    std::cout << sourceURL << " " << worker->name << " " << meta.size[0] << "x" << meta.size[1] << " " << worker->length << " bytes" << std::endl;

    return 0;
}