#include "../../lt.h"
#include <iostream>

using string = std::string;

#ifdef _WIN32
#include <direct.h>
#endif

void logFatal(lt::error err)
{
    std::cout << err << std::endl;
    exit(-1);
}

int main()
{
    lt::error err;
    lt::Client client;

    // Find first active input video source
    std::string sourceURL;
    std::vector<string> sources = {
        "lt100:/0/cvbs-in/0",
        "lt100:/0/svideo-in/0",
        "lt100:/0/dvi-in/0", "lt100:/0/dvi-in/1",
        "lt100:/0/sdi-in/0", "lt100:/0/sdi-in/1",
    };
    for (auto& source : sources) {
        lt::Input input;
        err = client.Get(source, input);
        if (!err.empty()) {
            logFatal(err);
        }
        if (input.video.signal == "locked") {
            sourceURL = source;
            break;
        }
    }

    // If no active input video source found, use a default one
    if (sourceURL == "") {
        printf("no active input video source found\n");
        sourceURL = "lt100:/0/sdi-in/0";
    }

    // Create worker
    err = client.Post(sourceURL + "/data", lt::ImageDataWorker { "image/yuv422" }, nullptr);
    if (!lt::ErrorIs(err, lt::ErrRedirect)) {
        logFatal("worker creation failed:" + err);
    }
    string workerURL = lt::RedirectLocation(err);

    // Fetch worker response
    lt::Worker worker;
    err = client.Get(workerURL, worker);
    if (!err.empty()) {
        logFatal(err);
    }

    // Check packet
    if (worker->packets.empty()) {
        logFatal("worker packet: not found");
    }
    lt::Packet packet = worker->packets[0];

    // Packet metadata
    lt::VideoMetadata meta = packet->meta.template get<lt::VideoMetadata>();

    // Print infos
    std::cout << sourceURL << " " << meta.size[0] << "x" << meta.size[1] << " " << packet->length << " bytes" << std::endl;

    // Create histogram
    int n = meta.size[0] * meta.size[1];
    unsigned char* data = (unsigned char*)packet->data;
    int h[8] = {};
    int i = 0;
    for (i = 0; i < n; i++) {
        h[data[i] / 32]++;
    }

	// Print histogram
    printf("\n");
    printf("  0.. 31: %.1f%%\n", 100 * float(h[0]) / float(n));
    printf(" 32.. 63: %.1f%%\n", 100 * float(h[1]) / float(n));
    printf(" 64.. 95: %.1f%%\n", 100 * float(h[2]) / float(n));
    printf(" 96..127: %.1f%%\n", 100 * float(h[3]) / float(n));
    printf("128..159: %.1f%%\n", 100 * float(h[4]) / float(n));
    printf("160..191: %.1f%%\n", 100 * float(h[5]) / float(n));
    printf("192..223: %.1f%%\n", 100 * float(h[6]) / float(n));
    printf("224..256: %.1f%%\n", 100 * float(h[7]) / float(n));

    return 0;
}