"""
Example: List LT devices present in the system and for each one fetch inputs status.

This example demonstrates how to:
    1. List all LT devices connected to the system.
    2. For each device, list all available input sources.
    3. Fetch and display the status of each input source.

Usage:
  	inputs.exe

Note:
    This code is intended for demonstration purposes. For production use,
    ensure proper error handling and resource cleanup.

© 2025 Enciris Technologies. All rights reserved.
"""

if __package__:
    from ... import lt
else:
    import sys
    sys.path.append("../..")
    import lt


board_inputs = {
    "lt311": ["sdi-in/0", "sdi-in/1", "sdi-in/2", "sdi-in/3"],
    "lt312": ["hdmi-in/0", "hdmi-in/1"],
    "lt313": ["sdi-in/0", "sdi-in/1", "hdmi-in/0"],
}

def main():
    client = lt.Client()

    # Try all possible LT devices (up to 2 boards in the system)
    for i in range(2):
        try:
            board = client.Get(f"lt310:/{i}")
        except Exception:
            # No more devices found
            print(f"Board {i}: not found")
            return

        print(f"Board {i}: {board['model']} ({board['sn']})")

        # Fetch inputs status
        for input_name in board_inputs.get(board["model"], []):
            input_url = f"lt310:/{i}/{input_name}"
            try:
                input_status = client.Get(input_url)
            except Exception as e:
                print(f"  Error getting input {input_name}: {e}")
                continue

            print(f"  Input {input_name}:")
            if input_status["video"]["signal"] == "none":
                print("    No signal")
                continue

            video = input_status["video"]
            print(f"    Video: {video['size'][0]}x{video['size'][1]} { 'i' if video['interlaced'] else 'p' } {video['framerate']:.2f}Hz {video['format']}")
            audio = input_status["audio"]
            print(f"    Audio: {audio['format']}, {audio['channels']} channels @ {audio['samplerate']}Hz")


if __name__ == "__main__":
    main()
