package main

//
// Example: Capture a snapshot from an LT device input
//
// This example demonstrates how to:
//      1. Connect to a given input source on an LT device.
//      2. Verify that the source has a locked video signal.
//      3. Create a worker to capture a snapshot in JPEG format.
//      4. Save the snapshot to the current working directory.
//      5. Print the file name, resolution, and file size.
//
// Usage:
//      still_capture.exe -source <sourceURL>
//
// Arguments:
//      -source, -s   URL of a valid LT device input, for example:
//                        lt310:/0/hdmi-in/0
//                        lt310:/0/sdi-in/1
//
// Note:
//      This code is intended for demonstration purposes. For production use,
//      ensure proper error handling and resource cleanup.
//
// © 2025 Enciris Technologies. All rights reserved.
//

import (
	"encoding/json"
	"errors"
	"flag"
	"fmt"
	"log"
	"os"

	lt "lt/client/go"
)

var argUsage = `
Arguments:
    -source <String>   URL of a valid LT device input (e.g. lt310:/0/hdmi-in/0)

This example verifies that the selected input source is active,
captures a snapshot as a JPEG image, and prints its file information.
`

func main() {
	// Command-line flags
	flag.Usage = func() {
		fmt.Printf("Usage:\n    %s -source <sourceURL>\n", os.Args[0])
		fmt.Print(argUsage)
		os.Exit(1)
	}

	var sourceURL string
	flag.StringVar(&sourceURL, "source", "", "Source URL (e.g. lt310:/0/hdmi-in/0)")
	flag.Parse()

	if sourceURL == "" {
		flag.Usage()
	}

	// Create LT client
	var client lt.Client
	defer client.Close()

	// -------------------------------------------------------------------------
	// Check if input selected is active
	// -------------------------------------------------------------------------
	var input lt.Input
	if err := client.Get(sourceURL, &input); err != nil {
		log.Fatal(err)
	}
	if input.Video.Signal != "locked" {
		log.Fatal("input selected is not locked")
	}

	// -------------------------------------------------------------------------
	// Take a snapshot of the selected input
	// -------------------------------------------------------------------------

	// Select an absolute path to a capture directory
	wd, err := os.Getwd() // Current working directory
	if err != nil {
		log.Fatal("working directory:", err)
	}

	// Create worker to capture the image
	err = client.Post(sourceURL+"/file", lt.ImageFileWorker{Media: "image/jpeg", Location: wd}, nil)
	if !errors.Is(err, lt.ErrRedirect) {
		log.Fatal("worker creation failed:", err)
	}
	workerURL := lt.RedirectLocation(err)

	// Fetch worker response
	var worker lt.Worker
	if err := client.Get(workerURL, &worker); err != nil {
		log.Fatal(err)
	}

	// Packet data
	if len(worker.Packets) == 0 {
		log.Fatal("worker packet: not found")
	}
	packet := worker.Packets[0]
	defer packet.Close() // Packet has to be manually released

	// -------------------------------------------------------------------------
	// Print file information
	// -------------------------------------------------------------------------

	// Packet metadata
	var meta lt.ImageMetadata
	if err := json.Unmarshal(packet.Meta, &meta); err != nil {
		log.Fatal("worker packet metadata:", err)
	}
	fmt.Printf("%s %s %dx%d %d bytes\n", sourceURL, worker.Name, meta.Size[0], meta.Size[1], worker.Length)
}
