package main

//
// Example: List LT devices present in the system and for each one fetch inputs status.
//
// This example demonstrates how to:
// 		1. List all LT devices connected to the system.
// 		2. For each device, list all available input sources.
// 		3. Fetch and display the status of each input source.
//
// Usage:
//   	inputs.exe
//
// Note:
//      This code is intended for demonstration purposes. For production use,
//      ensure proper error handling and resource cleanup.
//
// © 2025 Enciris Technologies. All rights reserved.
//

import (
	"fmt"
	lt "lt/client/go"
)

var boardInputs = map[string][]string{
	"lt311": {"sdi-in/0", "sdi-in/1", "sdi-in/2", "sdi-in/3"},
	"lt312": {"hdmi-in/0", "hdmi-in/1"},
	"lt313": {"sdi-in/0", "sdi-in/1", "hdmi-in/0"},
}

func main() {
	// Create LT client
	var client lt.Client
	defer client.Close()

	// -------------------------------------------------------------------------
	// Try all possible LT devices (up to 2 boards in the system)
	// -------------------------------------------------------------------------
	for i := range 2 {
		var board lt.Board
		if err := client.Get(fmt.Sprintf("lt310:/%d", i), &board); err != nil {
			// No more devices found
			fmt.Printf("Board %d: not found\n", i)
			continue
		}
		fmt.Printf("Board %d: %s (%d)\n", i, board.Model, board.SN)

		// Fetch inputs status
		for _, input := range boardInputs[board.Model] {
			var inputStatus lt.Input
			inputURL := fmt.Sprintf("lt310:/%d/%s", i, input)
			if err := client.Get(inputURL, &inputStatus); err != nil {
				fmt.Printf("  Error getting input %s: %v\n", input, err)
				continue
			}
			fmt.Printf("  Input %s:\n", input)
			if inputStatus.Video.Signal == "none" {
				fmt.Printf("    No signal\n")
				continue
			}
			fmt.Printf("    Video: %dx%d %s %.2fHz %s\n",
				inputStatus.Video.Size[0],
				inputStatus.Video.Size[1],
				map[bool]string{true: "i", false: "p"}[inputStatus.Video.Interlaced],
				inputStatus.Video.Framerate,
				inputStatus.Video.Format,
			)
			fmt.Printf("    Audio: %s, %d channels @ %dHz\n",
				inputStatus.Audio.Format,
				inputStatus.Audio.Channels,
				inputStatus.Audio.Samplerate,
			)
		}
	}
}
