package main

//
// Example: configure HDMI output and display a text overlay using the LT API client
//
// This example demonstrates how to:
//      1. Connect to an LT310 device and verify that a given input source is active.
//      2. Configure an HDMI output to use that input source.
//      3. Initialize a canvas and draw text on it.
//      4. Set the HDMI output overlay to display the canvas.
//
// Usage:
//      hdmi_out.exe -source <inputSource> [-text <overlayText>]
//
// Arguments:
//      -source   URL of a valid input on the LT device, for example:
//                    hdmi-in/0
//                    sdi-in/1
//
//      -text     Optional text to render on the video overlay.
//                Default: "Hello, World!"
//
// Note:
//      This code is intended for demonstration purposes. For production use,
//      ensure proper error handling and resource cleanup.
//
// © 2025 Enciris Technologies. All rights reserved.
//

import (
	"flag"
	"fmt"
	"log"
	"os"

	lt "lt/client/go"
)

var argUsage = `
Arguments:
    -source <string>   URL of a valid input on the LT device (e.g. hdmi-in/0)

    -text <string>     Optional text to render on the video overlay (default: "Hello, World!")

This example routes the selected input to HDMI output 0, creates a canvas with custom text,
and applies the canvas as an overlay on the HDMI output.
`

// Base path for the LT310 board : first board in the system
const boardBase = "lt310:/0/"

func main() {
	// Command-line flags
	flag.Usage = func() {
		fmt.Printf("Usage:\n    %s -source <sourceURL> -text <canvasText>\n", os.Args[0])
		fmt.Print(argUsage)
		os.Exit(1)
	}

	var sourceURL, canvasText string
	flag.StringVar(&sourceURL, "source", "", "Source URL (e.g. hdmi-in/0)")
	flag.StringVar(&sourceURL, "s", "", "Source URL (shorthand)")
	flag.StringVar(&canvasText, "text", "Hello, World!", "Text to display on the canvas")
	flag.StringVar(&canvasText, "t", "Hello, World!", "Text to display on the canvas (shorthand)")
	flag.Parse()

	if sourceURL == "" {
		flag.Usage()
	}

	// Create LT client
	var client lt.Client
	defer client.Close()

	// -------------------------------------------------------------------------
	// Check if input selected is active
	// -------------------------------------------------------------------------
	var input lt.Input
	if err := client.Get(boardBase+sourceURL, &input); err != nil {
		log.Fatal(err)
	}
	if input.Video.Signal != "locked" {
		log.Fatal("input selected is not locked")
	}

	// -------------------------------------------------------------------------
	// Set HDMI output source
	// -------------------------------------------------------------------------
	if err := client.Post(boardBase+"hdmi-out/0", lt.JSON{"source": sourceURL}, nil); err != nil {
		log.Fatal(err)
	}
	fmt.Printf("hdmi-out source set to: %s\n", sourceURL)

	// -------------------------------------------------------------------------
	// Get HDMI output parameters
	// -------------------------------------------------------------------------
	var output lt.Output
	if err := client.Get(boardBase+"hdmi-out/0", &output); err != nil {
		log.Fatal(err)
	}

	// -------------------------------------------------------------------------
	// Initialize the canvas with the same size as HDMI output
	// -------------------------------------------------------------------------
	if err := client.Post("lt310:/canvas/0/init", lt.CanvasInit{Size: output.Video.Size}, nil); err != nil {
		log.Fatal(err)
	}

	// -------------------------------------------------------------------------
	// Draw text on the canvas
	// -------------------------------------------------------------------------
	if err := client.Post("lt310:/canvas/0/text",
		lt.CanvasText{
			Text:     canvasText,
			FontSize: 200,
			Color:    [4]int{255, 255, 255, 255},
			Size:     output.Video.Size,
		}, nil); err != nil {
		log.Fatal(err)
	}

	/* Add your own drawing code here
	 * Images (PNG, JPEG, ...)
	 * Rectangles
	 * Lines
	 * Ellipses
	 * ...
	 */

	// -------------------------------------------------------------------------
	// Apply the canvas as an HDMI overlay
	// -------------------------------------------------------------------------
	if err := client.Post(boardBase+"hdmi-out/0", lt.JSON{"overlay": "canvas/0"}, nil); err != nil {
		log.Fatal(err)
	}

}
