/*
 * Example: List LT devices present in the system and for each one fetch inputs status.
 *
 * This example demonstrates how to:
 *      1. List all LT devices connected to the system.
 *      2. For each device, list all available input sources.
 *      3. Fetch and display the status of each input source.
 *
 * Usage:
 *      inputs.exe
 *
 * Note:
 *      This code is intended for demonstration purposes. For production use,
 *      ensure proper error handling and resource cleanup.
 *
 * © 2025 Enciris Technologies. All rights reserved.
 */

#include "../../lt.h"
#include <iostream>

#ifdef _WIN32
#include <direct.h>
#endif

// Map of board model -> available input names
const std::map<std::string, std::vector<std::string>> boardInputs = {
    {"lt311", {"sdi-in/0", "sdi-in/1", "sdi-in/2", "sdi-in/3"}},
    {"lt312", {"hdmi-in/0", "hdmi-in/1"}},
    {"lt313", {"sdi-in/0", "sdi-in/1", "hdmi-in/0"}},
};

/**
 * Main application entry point.
 */
int main()
{
    lt::Client client;
    lt::error err;

    // -------------------------------------------------------------------------
    // Try all possible LT devices (up to 2 boards in the system)
    // -------------------------------------------------------------------------
    for (int i = 0; i < 2; i++) {
        lt::Board board;
        err = client.Get("lt310:/" + std::to_string(i), board);
        if (!err.empty()) {
            std::cout << "Board " << i << ": not found\n";
            continue; // no more devices found
        }
        std::cout << "Board " << i << ": " << board.model << " (" << board.sn << ")\n";

        // ---------------------------------------------------------------------
        // List all inputs for this device
        // ---------------------------------------------------------------------
        // Find inputs for this board model
        auto it = boardInputs.find(board.model);
        if (it == boardInputs.end()) {
            std::cout << "  Unknown board model\n";
            continue;
        }

        // Fetch input status
        for (const auto &inputName : it->second) {
            lt::Input inputStatus;
            std::string inputURL = "lt310:/" + std::to_string(i) + "/" + inputName;

            err = client.Get(inputURL, inputStatus);
            if (!err.empty()) {
                std::cout << "  Error getting input " << inputName << ": " << err << "\n";
                continue;
            }
            std::cout << "  Input " << inputName << ":\n";

            if (inputStatus.video.signal == "none") {
                std::cout << "    No signal\n";
                continue;
            }

            std::printf("    Video: %dx%d %s %.2fHz %s\n",
                inputStatus.video.size[0],
                inputStatus.video.size[1],
                inputStatus.video.interlaced ? "i" : "p",
                inputStatus.video.framerate,
                inputStatus.video.format.c_str());

            std::printf("    Audio: %s, %d channels @ %dHz\n",
                inputStatus.audio.format.c_str(),
                inputStatus.audio.channels,
                inputStatus.audio.samplerate);
        }
    }

    return 0;
}